"""Qualitymaxxer

Made by AI 

Scans a source audio file (preferably FLAC) and encodes several OGG Vorbis
files at different quality settings, measures their average bitrates, and
exports the file with the highest bitrate not exceeding a configurable
maximum (default 208 kbps) as "qualitymaxxed audio.ogg".


Requires ffmpeg and ffprobe available on PATH.
"""

import sys
import subprocess
import time
import os
import pathlib
import argparse
import tempfile
import shutil
import math
try:
	# tkinter is part of stdlib but may not be available in some minimal Python installs
	import tkinter as tk
	from tkinter import filedialog
except Exception:
	tk = None
	filedialog = None
try:
	import msvcrt
except Exception:
	msvcrt = None


def check_tool(name):
	try:
		subprocess.run([name, "-version"], stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL, check=True)
		return True
	except Exception:
		return False


def ffprobe_get_metrics(path):
	"""Return (bitrate_kbps (float or None), duration_seconds (float or None)).
	Attempts format-level bit_rate first, falls back to computing from file size / duration.
	"""
	try:
		# ask ffprobe for duration only; some builds report a bad/zero format bit_rate
		cmd = ["ffprobe", "-v", "error", "-show_entries", "format=duration", "-of", "default=noprint_wrappers=1:nokey=1", path]
		out = subprocess.check_output(cmd, stderr=subprocess.DEVNULL).decode().strip().splitlines()
		duration = None
		if len(out) >= 1 and out[0].strip() != "N/A" and out[0].strip() != "":
			try:
				duration = float(out[0].strip())
			except Exception:
				duration = None

		if duration and duration > 0:
			size_bytes = os.path.getsize(path)
			bit_rate = (size_bytes * 8.0) / duration
			bitrate_kbps = bit_rate / 1000.0
			return bitrate_kbps, duration
		return None, duration
	except subprocess.CalledProcessError:
		return None, None


def encode_vorbis(input_path, output_path, quality):
	"""Encode input_path to output_path using libvorbis quality (0.0-10.0).
	Returns True on success.
	"""
	# Map only audio streams, copy global metadata from input, and exclude
	# any video/attached picture streams so embedded images are not included.
	# Use first audio stream (0:a:0) which is typical; mapping all audio
	# streams into an OGG container is uncommon.
	cmd = [
		"ffmpeg", "-y", "-hide_banner", "-loglevel", "error",
		"-i", input_path,
		"-map", "0:a:0",
		"-map_metadata", "0",
		"-map", "-0:v",
		"-c:a", "libvorbis",
		"-q:a", str(quality),
		output_path,
	]
	try:
		subprocess.run(cmd, check=True)
		return True
	except subprocess.CalledProcessError:
		return False


def find_best_candidate(candidates, max_kbps):
	"""Given list of tuples (path, bitrate_kbps), choose best per rules:
	- prefer highest bitrate <= max_kbps
	- if none <= max_kbps, choose one with bitrate closest to max_kbps (either side)
	Returns index in candidates or None.
	"""
	valid = [(i, b) for i, (_, b) in enumerate(candidates) if b is not None and b <= max_kbps]
	if valid:
		# pick max bitrate among valid
		best = max(valid, key=lambda x: x[1])
		return best[0]
	# otherwise pick candidate with minimal absolute difference
	diffs = [(i, abs((b if b is not None else float('inf')) - max_kbps)) for i, (_, b) in enumerate(candidates)]
	if not diffs:
		return None
	best = min(diffs, key=lambda x: x[1])
	return best[0]


def parse_quality_list(start, end, step):
	if step <= 0:
		raise ValueError("step must be positive")
	vals = []
	# handle both increasing and decreasing ranges
	if abs(end - start) < 1e-12:
		return [round(start, 2)]
	sign = 1 if end > start else -1
	cur = start
	# use a safe loop with a max iterations guard
	max_iters = 10000
	it = 0
	cmp = (lambda a, b: a <= b + 1e-9) if sign == 1 else (lambda a, b: a >= b - 1e-9)
	while cmp(cur, end) and it < max_iters:
		vals.append(round(cur, 2))
		cur += sign * step
		it += 1
	# ensure end included
	if not vals or abs(vals[-1] - end) > 1e-6:
		vals.append(round(end, 2))
	# preserve order (ascending or descending) and uniqueness
	seen = []
	for v in vals:
		if v not in seen:
			seen.append(v)
	return seen


def main(argv=None):
	if argv is None:
		argv = sys.argv[1:]

	ap = argparse.ArgumentParser(description="Encode input audio into multiple OGG Vorbis qualities and pick the best under a max bitrate.")
	ap.add_argument("input", nargs="?", help="Input audio file (prefer .flac)")
	ap.add_argument("--start", type=float, default=6.0, help="start quality (default 6.0)")
	ap.add_argument("--end", type=float, default=7.0, help="end quality (default 7.0)")
	ap.add_argument("--step", type=float, default=0.1, help="step between qualities (default 0.1)")
	ap.add_argument("--max-kbps", type=float, default=208.0, help="maximum allowed average bitrate in kbps (default 208)")
	ap.add_argument("--output", default="qualitymaxxed audio.ogg", help="output filename (default 'qualitymaxxed audio.ogg')")
	ap.add_argument("--keep-temp", action="store_true", help="keep temporary encoded files for inspection")
	args = ap.parse_args(argv)

	# if no input, open a Tk file dialog if available otherwise fallback to text prompt
	if not args.input:
		print("Please select audio file...")
		time.sleep(1.5)
		if tk is not None and filedialog is not None:
			try:
				root = tk.Tk()
				root.withdraw()
				selected = filedialog.askopenfilename(title="Select audio file to qualitymaxx", filetypes=[("Audio files", "*.flac *.wav *.ogg *.m4a"), ("All files", "*")])
				root.destroy()
				if not selected:
					print("File dialog closed or failed to select a file.")
					time.sleep(1.5)
					return 1
				args.input = selected
			except Exception:
				print("File dialog failed to open.")
				time.sleep(1.5)
				# fallback to console prompt
				args.input = input("Input file path: ").strip('"')
		else:
			print("File dialog not available; please type path.")
			time.sleep(1.5)
			args.input = input("Input file path: ").strip('"')

	input_path = os.path.abspath(args.input)
	if not os.path.exists(input_path):
		print(f"Input file not found: {input_path}")
		return 2

	if not check_tool("ffmpeg") or not check_tool("ffprobe"):
		print("ffmpeg and ffprobe must be installed and on PATH. See https://ffmpeg.org/")
		return 3

	qualities = parse_quality_list(args.start, args.end, args.step)

	# create a predictable temp dir name; if it exists, append an index
	tmp_base = os.path.join(tempfile.gettempdir(), "qualitymaxxer")
	if not os.path.exists(tmp_base):
		work_dir = tmp_base
		os.makedirs(work_dir, exist_ok=True)
	else:
		# find an indexed directory name
		i = 1
		while True:
			candidate = f"{tmp_base} ({i})"
			if not os.path.exists(candidate):
				work_dir = candidate
				os.makedirs(work_dir, exist_ok=True)
				break
			i += 1

	candidates = []
	created_files = []
	print(f"Trying qualities: {qualities}")

	# Determine iteration direction (qualities list preserves order)
	# We'll iterate through qualities list and stop early if a candidate exceeds max_kbps.
	# If the very first tried quality already exceeds max_kbps, do a downward search from that starting quality until bitrate < max_kbps or reach 0.0.
	encoded_set = set()
	def try_encode(q):
		# avoid re-encoding same quality
		if q in encoded_set:
			return None, None, None
		encoded_set.add(q)
		out_name = f"q{str(q).replace('.','_')}.ogg"
		out_path = os.path.join(work_dir, out_name)
		t0 = time.time()
		ok = encode_vorbis(input_path, out_path, q)
		t1 = time.time()
		if not ok:
			print(f"Encoding failed for quality={q}")
			# ensure we don't leave a partial file recorded
			if os.path.exists(out_path):
				created_files.append(out_path)
			return out_path, None, t1 - t0
		# only record files that actually exist on disk
		if os.path.exists(out_path):
			created_files.append(out_path)
		bitrate_kbps, duration = ffprobe_get_metrics(out_path)
		print(f"  quality={q} -> {bitrate_kbps if bitrate_kbps is not None else float('nan'):.1f} kbps (encode {t1-t0:.1f}s)")
		return out_path, bitrate_kbps, t1 - t0

	# iterate forward through qualities
	stopped_early = False
	for idx, q in enumerate(qualities):
		out_path, bitrate_kbps, _ = try_encode(q)
		candidates.append((out_path, bitrate_kbps))
		# if bitrate is unknown, continue trying next
		if bitrate_kbps is None:
			continue
		if bitrate_kbps > args.max_kbps:
			# if this is the first tried quality, start a downward search
			if idx == 0:
				cur = q - args.step
				# search downwards until bitrate < max_kbps or we hit 0.0
				while cur >= 0.0:
					out_path, bitrate_kbps, _ = try_encode(round(cur, 2))
					candidates.append((out_path, bitrate_kbps))
					if bitrate_kbps is not None and bitrate_kbps < args.max_kbps:
						break
					cur -= args.step
			# stop trying higher qualities
			stopped_early = True
			break

	if not candidates:
		print("No successful encodes produced.")
		return 4

	best_idx = find_best_candidate(candidates, args.max_kbps)
	if best_idx is None:
		print("Couldn't pick a best candidate.")
		return 5

	best_path, best_bitrate = candidates[best_idx]
	# Ask the user to choose an output filename (save-as) showing existing .ogg files
	if tk is not None and filedialog is not None:
		try:
			root = tk.Tk()
			root.withdraw()
			initial_dir = os.path.dirname(os.path.abspath(input_path))
			initial_file = args.output
			save_path = filedialog.asksaveasfilename(title="Save exported OGG as", defaultextension=".ogg", filetypes=[("OGG files", "*.ogg"), ("All files", "*")], initialdir=initial_dir, initialfile=initial_file)
			root.destroy()
			if save_path:
				desired = save_path
			else:
				# cancelled: fall back to input file directory
				dest_dir = initial_dir
				desired = os.path.join(dest_dir, args.output)
		except Exception:
			dest_dir = os.path.dirname(os.path.abspath(input_path))
			desired = os.path.join(dest_dir, args.output)
	else:
		dest_dir = os.path.dirname(os.path.abspath(input_path))
		desired = os.path.join(dest_dir, args.output)

	def unique_path(path):
		# If path doesn't exist, return as-is. Otherwise append " (n)" before extension
		if not os.path.exists(path):
			return path
		base, ext = os.path.splitext(path)
		i = 1
		while True:
			candidate = f"{base} ({i}){ext}"
			if not os.path.exists(candidate):
				return candidate
			i += 1

	dest_path = unique_path(desired)
	shutil.copyfile(best_path, dest_path)
	print(f"Selected: {os.path.basename(best_path)} with {best_bitrate:.1f} kbps")
	print(f"Exported to: {dest_path}")

	if args.keep_temp:
		print(f"Temporary files left in: {work_dir}")
	else:
		# Safely remove only the files we created and only if no unexpected files exist.
		try:
			present = set(os.listdir(work_dir))
			expected = set(os.path.basename(p) for p in created_files)
			# If there are unexpected files, refuse to delete anything
			unexpected = present - expected
			if unexpected:
				print("Refusing to delete temporary directory because it contains unexpected files:")
				for u in unexpected:
					print("  ", u)
				print(f"Temporary files left in: {work_dir}")
			else:
				# delete only the files we created
				for p in created_files:
					try:
						if os.path.exists(p):
							os.remove(p)
					except Exception:
						print(f"Failed to remove temp file: {p}")
				# directory must be empty before removal
				if not os.listdir(work_dir):
					os.rmdir(work_dir)
				else:
					print(f"Temporary directory not empty after removing known files; left in place: {work_dir}")
		except Exception as e:
			print(f"Error during temporary cleanup: {e}")

	print("Best audio candidate selected and exported.")
	try:
		if msvcrt is not None:
			print("Press any key to exit...")
			msvcrt.getch()
		else:
			input("Press Enter to exit...")
	except Exception:
		pass

	return 0


if __name__ == "__main__":
	sys.exit(main())
